module pickRGB


//	**************************************************************************************************
//
//	This program creates a dialog that allows a user to create a RGB colour.
//
//	The program has been written in Clean 1.3.1 and uses the Clean Standard Object I/O library 1.1
//	
//	**************************************************************************************************


import StdEnv, StdIO


::	NoState
	=	NoState


Start :: *World -> *World
Start world
	=	startIO NoState NoState [initIO] [] world

initIO :: (PSt .l .p) -> PSt .l .p
initIO ps
	#	(rgbid,ps)	= accPIO openR2Id ps
	#	(wid,  ps)	= accPIO openId   ps
	#	(ids,  ps)	= accPIO (openIds 7) ps
		wdef		= Dialog "Pick a colour"
						(	ColourPickControl wid rgbid ids initrgb
						)
						[	WindowId wid
						]
	#	(error,ps)	= openDialog undef wdef ps
	|	error<>NoError
		=	abort "pickRGB could not open window."
	#	mdef		= Menu "PickRGB"
						(	MenuItem "MinRGB" [MenuFunction (noLS (set rgbid BlackRGB)),MenuShortKey 'n']
						:+:	MenuItem "MaxRGB" [MenuFunction (noLS (set rgbid WhiteRGB)),MenuShortKey 'm']
						:+:	MenuSeparator     []
						:+:	MenuItem "Quit"   [MenuFunction (noLS closeProcess),        MenuShortKey 'q']
						)
						[]
	#	(error,ps)	= openMenu undef mdef ps
	|	error<>NoError
		=	abort "pickRGB could not open menu."
	|	otherwise
		=	ps
where
	initrgb			= {r=MaxRGB,g=MaxRGB,b=MaxRGB}
	
	set rid rgb ps	= snd (syncSend2 rid (InSet rgb) ps)


/*	The definition of the text-slider component:	*/

::	RGBPickControl ls ps
	:==	:+: SliderControl TextControl ls ps

RGBPickControl :: RGBColour Id (String,Id,Id) Id (RGBColour->Int) (Int->RGBColour->RGBColour)
	-> RGBPickControl RGBColour (PSt .l .p)
RGBPickControl rgb wid (text,sid,tid) did get set
	=	  SliderControl Horizontal length sliderstate slideraction
													[ControlId sid,ControlPos (Left,zero)]
	  :+: TextControl   (ColourText text (get rgb))	[ControlId tid]
where
	length		= MaxRGB-MinRGB+1
	sliderstate	= {sliderMin=MinRGB, sliderMax=MaxRGB, sliderThumb=get rgb}
	
	slideraction :: SliderMove (RGBColour,PSt .l .p) -> (RGBColour,PSt .l .p)
	slideraction move (rgb,ps)
		=	(newrgb,newps)
	where
		y		= case move of
					SliderIncSmall	-> min (get rgb+1 ) MaxRGB

					SliderDecSmall	-> max (get rgb-1 ) MinRGB
					SliderIncLarge	-> min (get rgb+10) MaxRGB
					SliderDecLarge	-> max (get rgb-10) MinRGB
					SliderThumb x	-> x
		newrgb	= set y rgb
		newps	= appPIO (	setWindow		wid
						 [	setSliderThumbs [(sid,y)]
						 ,	setControlTexts [(tid,ColourText text y)]
						 ,	SetColourBox	did newrgb
						 ])	ps
	
ColourText :: String Int -> String
ColourText text x
	=	text+++" "+++toString x



/*	The definition of a colour box:		*/

::	ColourBoxControl ls ps
	:==	CustomControl ls ps

ColourBoxControl :: RGBColour Id -> ColourBoxControl .ls .ps
ColourBoxControl rgb did
	=	CustomControl {w=40,h=40} (ColourBoxLook rgb) [ControlId did]

ColourBoxLook :: RGBColour SelectState UpdateState -> *Picture -> *Picture
ColourBoxLook colour _ {newFrame}
	= seq	[	setPenColour (RGB colour),	fill newFrame
			,	setPenColour Black,			draw newFrame
			]

SetColourBox :: Id RGBColour *WState -> *WState
SetColourBox id rgb wstate
	=	setControlLooks [(id,True,ColourBoxLook rgb)] wstate



/*	The definition of the RGB access control:	*/

::	In		=	InGet				| InSet RGBColour
::	Out		=	OutGet RGBColour	| OutSet
::	RGBId	:==	R2Id In Out

::	ColourPickAccess ps	:==	Receiver2 In Out RGBColour ps

ColourPickAccess :: RGBId Id [(String,Id,Id)] Id -> ColourPickAccess (PSt .l .p)
ColourPickAccess rid wid rgbpicks did
	=	Receiver2 rid accessRGB []
where
	accessRGB :: In (RGBColour,PSt .l .p) -> (Out,(RGBColour,PSt .l .p))
	accessRGB InGet (rgb,ps)
		=	(OutGet rgb,(rgb,ps))
	accessRGB (InSet rgb=:{r,g,b}) (_,ps=:{io})
		#	io	= setWindow wid [SetColourBox    did rgb
								,setSliderThumbs (map (\(y,(_,sid,_))->(sid,y)) settings)
								,setControlTexts (map (\(y,(text,_,tid))->(tid,ColourText text y)) settings)
								]	io
		=	(OutSet,(rgb,{ps & io=io}))
	where
		settings	= zip2 [r,g,b] rgbpicks



/*	The definition of the assembled colour picking control:	*/

::	ColourPickControl ls ps
/*	:==	:+:	(CompoundControl (ListLS RGBPickControl))
		:+:	ColourBoxControl
			ColourPickAccess
			ls	ps
*/	:==	NewLS
		(	:+:	(CompoundControl (ListLS (:+: SliderControl TextControl)))
		(	:+:	CustomControl
				(Receiver2 In Out)
		))	ls	ps

ColourPickControl :: Id RGBId [Id] RGBColour -> ColourPickControl .ls (PSt .l .p)
ColourPickControl wid rgbid ids initrgb
	=	{	newLS	= initrgb
		,	newDef	= CompoundControl
						(	ListLS
						[	RGBPickControl initrgb wid rpicks did (\rgb->rgb.r) (\x rgb->{rgb&r=x})
						,	RGBPickControl initrgb wid gpicks did (\rgb->rgb.g) (\x rgb->{rgb&g=x})
						,	RGBPickControl initrgb wid bpicks did (\rgb->rgb.b) (\x rgb->{rgb&b=x})
						])	[]
						:+: ColourBoxControl initrgb did
				:+:	ColourPickAccess rgbid wid [rpicks,gpicks,bpicks] did
		}
where	
	[rid,rtid,gid,gtid,bid,btid,did:_]	= ids
	(rtext,gtext,btext)					= ("Red","Green","Blue")
	(rpicks,gpicks,bpicks)				= ((rtext,rid,rtid),(gtext,gid,gtid),(btext,bid,btid))
